/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <NetworkManager.h>
#include <nm-device.h>
#include <nm-connection.h>
#include <nm-setting-connection.h>
#include <nm-setting-ip4-config.h>
#include "nmn-network-item.h"
#include "nmn-connection-details.h"
#include "nma-gconf-connection.h"

#define DBUS_TYPE_G_MAP_OF_VARIANT (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE))
#define DBUS_TYPE_G_MAP_OF_MAP_OF_VARIANT (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, DBUS_TYPE_G_MAP_OF_VARIANT))

G_DEFINE_TYPE (NmnNetworkItem, nmn_network_item, NMN_TYPE_ITEM)

enum {
    PROP_0,
    PROP_NM_DATA,
    PROP_DEVICE,
    PROP_CONNECTION,
    PROP_AC,

    LAST_PROP
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_NETWORK_ITEM, NmnNetworkItemPrivate))

typedef struct {
    NmnNMData *nm_data;
    NMDevice *device;
    NMExportedConnection *connection;
    NMActiveConnection *ac;

    gulong connection_updated_id;
    gulong connection_removed_id;
    gulong secrets_requested_id;
    gulong ac_state_changed_id;

    gboolean disposed;
} NmnNetworkItemPrivate;

NmnNMData *
nmn_network_item_get_nm_data (NmnNetworkItem *self)
{
    g_return_val_if_fail (NMN_IS_NETWORK_ITEM (self), NULL);

    return GET_PRIVATE (self)->nm_data;
}

NMDevice *
nmn_network_item_get_device (NmnNetworkItem *self)
{
    g_return_val_if_fail (NMN_IS_NETWORK_ITEM (self), NULL);

    return GET_PRIVATE (self)->device;
}

NMActiveConnection *
nmn_network_item_get_active_connection (NmnNetworkItem *self)
{
    g_return_val_if_fail (NMN_IS_NETWORK_ITEM (self), NULL);

    return GET_PRIVATE (self)->ac;
}

static void
ac_state_changed (NMActiveConnection *ac,
                  GParamSpec *pspec,
                  gpointer user_data)
{
    NmnItemStatus status;

    if (ac) {
        switch (nm_active_connection_get_state (ac)) {
        case NM_ACTIVE_CONNECTION_STATE_ACTIVATED:
            status = NMN_ITEM_STATUS_CONNECTED;
            break;
        case NM_ACTIVE_CONNECTION_STATE_ACTIVATING:
            status = NMN_ITEM_STATUS_CONNECTING;
            break;
        default:
            status = NMN_ITEM_STATUS_DISCONNECTED;
            break;
        }
    } else
        status = NMN_ITEM_STATUS_DISCONNECTED;

    nmn_item_set_status (NMN_ITEM (user_data), status);
    nmn_item_priority_changed (NMN_ITEM (user_data));
}

void
nmn_network_item_set_active_connection (NmnNetworkItem *self,
                                        NMActiveConnection *ac)
{
    NmnNetworkItemPrivate *priv;

    g_return_if_fail (NMN_IS_NETWORK_ITEM (self));

    priv = GET_PRIVATE (self);
    if (priv->ac) {
        g_signal_handler_disconnect (priv->ac, priv->ac_state_changed_id);
        g_object_unref (priv->ac);
    }

    if (ac) {
        priv->ac = g_object_ref (ac);
        priv->ac_state_changed_id = g_signal_connect (ac, "notify::" NM_ACTIVE_CONNECTION_STATE,
                                                      G_CALLBACK (ac_state_changed),
                                                      self);
    } else {
        priv->ac = NULL;
        priv->ac_state_changed_id = 0;
    }

    ac_state_changed (ac, NULL, self);
}

static void
update_item (NmnNetworkItem *self)
{
    NmnNetworkItemPrivate *priv = GET_PRIVATE (self);
    NmnItem *item = NMN_ITEM (self);
    NMConnection *wrapped;
    NMSettingConnection *s_con;

    g_assert (priv->connection);

    wrapped = nm_exported_connection_get_connection (priv->connection);
    s_con = NM_SETTING_CONNECTION (nm_connection_get_setting (wrapped, NM_TYPE_SETTING_CONNECTION));

    nmn_item_set_name (item, nm_setting_connection_get_id (s_con));
}

static void
updated (NMExportedConnection *connection,
         GHashTable *settings,
         gpointer user_data)
{
    update_item (NMN_NETWORK_ITEM (user_data));
}

static void
removed (NMExportedConnection *connection, gpointer user_data)
{
    nmn_item_remove_request (NMN_ITEM (user_data));
}

static void
connection_secrets_requested_cb (NMExportedConnection *connection,
                                 const char *setting_name,
                                 const char **hints,
                                 gboolean ask_user,
                                 DBusGMethodInvocation *context,
                                 gpointer user_data)
{
    NmnNetworkItem *self = NMN_NETWORK_ITEM (user_data);

    if (NMN_NETWORK_ITEM_GET_CLASS (self)->secrets_requested)
        NMN_NETWORK_ITEM_GET_CLASS (self)->secrets_requested (self, setting_name, hints, ask_user, context);
    else {
        GError *error = NULL;

        g_set_error_literal (&error, NM_SETTINGS_ERROR, NM_SETTINGS_ERROR_SECRETS_UNAVAILABLE,
                             G_STRLOC "no secrets found");

        g_warning ("%s", error->message);
        dbus_g_method_return_error (context, error);
        g_error_free (error);
    }
}

static void
connect_cb (gpointer user_data,
            const char *object_path,
            GError *error)
{
    /* FIXME: Report the error somewhere */
}

static void
connect (NmnItem *item)
{
    NmnNetworkItemPrivate *priv = GET_PRIVATE (item);
    NMConnection *wrapped;
    const char *path;
    const char *service_name;
    const char *specific_object;
    NMConnectionScope scope;

    if (!priv->connection) {
        g_warning ("Connection not set");
        return;
    }

    wrapped = nm_exported_connection_get_connection (priv->connection);
    path = nm_connection_get_path (wrapped);
    scope = nm_connection_get_scope (wrapped);

    service_name = (scope == NM_CONNECTION_SCOPE_USER) ?
        NM_DBUS_SERVICE_USER_SETTINGS : NM_DBUS_SERVICE_SYSTEM_SETTINGS;

    if (NMN_NETWORK_ITEM_GET_CLASS (item)->get_specific_object)
        specific_object = NMN_NETWORK_ITEM_GET_CLASS (item)->get_specific_object (NMN_NETWORK_ITEM (item));
    else
        specific_object = "/";

    nm_client_activate_connection (NM_CLIENT (priv->nm_data),
                                   service_name,
                                   path,
                                   priv->device,
                                   specific_object,
                                   connect_cb,
                                   NULL);
}

static void
disconnect (NmnItem *item)
{
    NmnNetworkItemPrivate *priv = GET_PRIVATE (item);

    if (!priv->ac) {
        g_warning ("Can not disconnect, the item is not active?");
        return;
    }

    nm_client_deactivate_connection (NM_CLIENT (priv->nm_data), priv->ac);
}

NMExportedConnection *
nmn_network_item_get_connection (NmnNetworkItem *self)
{
    g_return_val_if_fail (NMN_IS_NETWORK_ITEM (self), NULL);

    return GET_PRIVATE (self)->connection;
}

void
nmn_network_item_set_connection (NmnNetworkItem *self,
                                 NMExportedConnection *connection)
{
    NmnNetworkItemPrivate *priv;

    g_return_if_fail (NMN_IS_NETWORK_ITEM (self));

    priv = GET_PRIVATE (self);

    if (priv->connection) {
        g_signal_handler_disconnect (priv->connection, priv->connection_updated_id);
        g_signal_handler_disconnect (priv->connection, priv->connection_removed_id);

        if (priv->secrets_requested_id)
            g_signal_handler_disconnect (priv->connection, priv->secrets_requested_id);

        g_object_unref (priv->connection);
    }

    if (connection) {
        priv->connection = g_object_ref (connection);
        priv->connection_updated_id = g_signal_connect (connection, "updated", G_CALLBACK (updated), self);
        priv->connection_removed_id = g_signal_connect (connection, "removed", G_CALLBACK (removed), self);

        if (NMA_IS_GCONF_CONNECTION (connection))
            priv->secrets_requested_id = g_signal_connect (connection, "new-secrets-requested",
                                                           G_CALLBACK (connection_secrets_requested_cb), self);
        else
            priv->secrets_requested_id = 0;

        update_item (self);
    } else {
        priv->connection = NULL;
        priv->connection_updated_id = 0;
    }
}

static void
item_delete (NmnItem *item)
{
    NMExportedConnection *exported = nmn_network_item_get_connection (NMN_NETWORK_ITEM (item));

    if (exported)
        nm_exported_connection_delete (exported, NULL);
}

static guint
get_priority (NmnItem *item)
{
    NMActiveConnection *ac;
    guint priority = 0;

    ac = nmn_network_item_get_active_connection (NMN_NETWORK_ITEM (item));
    if (ac) {
        priority = 90;

        if (nm_active_connection_get_default (ac))
            priority += 10;
    }

    return priority;
}

static GtkWidget *
create_advanced_information (NmnItem *item)
{
    NmnNetworkItemPrivate *priv = GET_PRIVATE (item);
    GtkWidget *details;

    details = nmn_connection_details_new (nmn_item_get_status (item) == NMN_ITEM_STATUS_DISCONNECTED);

    if (priv->connection) {
        NMConnection *connection;
        NMSettingIP4Config *setting;

        connection = nm_exported_connection_get_connection (priv->connection);
        setting = (NMSettingIP4Config *) nm_connection_get_setting (connection, NM_TYPE_SETTING_IP4_CONFIG);
        if (setting)
            nmn_connection_details_set_setting (NMN_CONNECTION_DETAILS (details), setting);
    }

    if (nm_device_get_state (priv->device) == NM_DEVICE_STATE_ACTIVATED) {
        NMIP4Config *config;

        config = nm_device_get_ip4_config (priv->device);
        if (config)
            nmn_connection_details_set_config (NMN_CONNECTION_DETAILS (details), config);
    }

    return details;
}

static void
nmn_network_item_init (NmnNetworkItem *item)
{
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnNetworkItemPrivate *priv;

    object = G_OBJECT_CLASS (nmn_network_item_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    if (!priv->nm_data || !priv->device) {
        g_warning ("%s: Missing constructor arguments", G_STRLOC);
        g_object_unref (object);
        return NULL;
    }

    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    NmnNetworkItemPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        /* Construct only */
        priv->nm_data = g_value_dup_object (value);
        break;
    case PROP_DEVICE:
        /* Construct only */
        priv->device = g_value_dup_object (value);
        break;
    case PROP_CONNECTION:
        nmn_network_item_set_connection (NMN_NETWORK_ITEM (object),
                                         (NMExportedConnection *) g_value_get_object (value));
        break;
    case PROP_AC:
        nmn_network_item_set_active_connection (NMN_NETWORK_ITEM (object),
                                                (NMActiveConnection *) g_value_get_object (value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    NmnNetworkItemPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_NM_DATA:
        g_value_set_object (value, priv->nm_data);
        break;
    case PROP_DEVICE:
        g_value_set_object (value, priv->device);
        break;
    case PROP_CONNECTION:
        g_value_set_object (value, priv->connection);
        break;
    case PROP_AC:
        g_value_set_object (value, priv->ac);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
dispose (GObject *object)
{
    NmnNetworkItem *self = NMN_NETWORK_ITEM (object);
    NmnNetworkItemPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    nmn_network_item_set_connection (self, NULL);

    if (priv->device)
        g_object_unref (priv->device);

    nmn_network_item_set_connection (self, NULL);

    if (priv->nm_data)
        g_object_unref (priv->nm_data);

    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_network_item_parent_class)->dispose (object);
}

static void
nmn_network_item_class_init (NmnNetworkItemClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    NmnItemClass *item_class = NMN_ITEM_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnNetworkItemPrivate));

    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->dispose = dispose;

    item_class->connect = connect;
    item_class->disconnect = disconnect;
    item_class->delete = item_delete;
    item_class->get_priority = get_priority;
    item_class->create_advanced_information = create_advanced_information;

    /* properties */
    g_object_class_install_property
        (object_class, PROP_NM_DATA,
         g_param_spec_object (NMN_NETWORK_ITEM_NM_DATA,
                              "NmnNMData",
                              "NmnNMData",
                              NMN_TYPE_NM_DATA,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    g_object_class_install_property
        (object_class, PROP_DEVICE,
         g_param_spec_object (NMN_NETWORK_ITEM_DEVICE,
                              "NMDevice",
                              "NMDevice",
                              NM_TYPE_DEVICE,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    g_object_class_install_property
        (object_class, PROP_CONNECTION,
         g_param_spec_object (NMN_NETWORK_ITEM_CONNECTION,
                              "NMExportedConnection",
                              "NMExportedConnection",
                              NM_TYPE_EXPORTED_CONNECTION,
                              G_PARAM_READWRITE));

    g_object_class_install_property
        (object_class, PROP_AC,
         g_param_spec_object (NMN_NETWORK_ITEM_AC,
                              "NMActiveConnection",
                              "NMActiveConnection",
                              NM_TYPE_ACTIVE_CONNECTION,
                              G_PARAM_READWRITE));
}
