/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <NetworkManager.h>
#include <nm-connection.h>
#include <nm-setting-connection.h>
#include "nmn-ethernet-item.h"

G_DEFINE_TYPE (NmnEthernetItem, nmn_ethernet_item, NMN_TYPE_NETWORK_ITEM)

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_ETHERNET_ITEM, NmnEthernetItemPrivate))

typedef struct {
    gulong state_changed_id;
    gboolean disposed;
} NmnEthernetItemPrivate;

static void
state_changed (NMDevice *device,
               NMDeviceState new_state,
               NMDeviceState old_state,
               NMDeviceStateReason reason,
               gpointer user_data)
{
    const char *icon;

    switch (new_state) {
    case NM_DEVICE_STATE_ACTIVATED:
        icon = "nm-device-wired";
        break;
    default:
        icon = "nm-no-connection";
        break;
    }

    nmn_item_set_icon (NMN_ITEM (user_data), icon);
}

GtkWidget *
nmn_ethernet_item_new (NmnNMData *nm_data,
                       NMDeviceEthernet *device)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (nm_data), NULL);
    g_return_val_if_fail (NM_IS_DEVICE_ETHERNET (device), NULL);

    return GTK_WIDGET (g_object_new (NMN_TYPE_ETHERNET_ITEM,
                                     NMN_NETWORK_ITEM_NM_DATA, nm_data,
                                     NMN_NETWORK_ITEM_DEVICE, device,
                                     NULL));
}

static gboolean
update_autoconnect (NmnNetworkItem *item, gboolean connect_automatically)
{
    NMExportedConnection *exported;
    NMConnection *wrapped;
    NMSettingConnection *s_con;
    GHashTable *new_settings;
    GError *error = NULL;

    exported = nmn_network_item_get_connection (item);
    wrapped = nm_exported_connection_get_connection (exported);
    s_con = NM_SETTING_CONNECTION (nm_connection_get_setting (wrapped, NM_TYPE_SETTING_CONNECTION));

    if (nm_setting_connection_get_autoconnect (s_con) == connect_automatically)
        return FALSE;

    g_object_set (s_con, NM_SETTING_CONNECTION_AUTOCONNECT, connect_automatically, NULL);
    new_settings = nm_connection_to_hash (wrapped);

    if (!nm_exported_connection_update (exported, new_settings, &error)) {
        if (error) {
            g_warning ("Couldn't update ethernet connection: %s", error->message);
            g_error_free (error);
        } else
            g_warning ("Couldn't update ethernet connection: no details");
    }

    g_hash_table_unref (new_settings);

    return TRUE;
}

static void
connect (NmnItem *item)
{
    update_autoconnect (NMN_NETWORK_ITEM (item), TRUE);
    NMN_ITEM_CLASS (nmn_ethernet_item_parent_class)->connect (item);
}

static void
disconnect (NmnItem *item)
{
    /* Turn off autoconnect, otherwise it would reconnect right back. */
    update_autoconnect (NMN_NETWORK_ITEM (item), FALSE);
    NMN_ITEM_CLASS (nmn_ethernet_item_parent_class)->disconnect (item);
}

static guint
get_priority (NmnItem *item)
{
    return NMN_ITEM_CLASS (nmn_ethernet_item_parent_class)->get_priority (item) + 50;
}

static void
nmn_ethernet_item_init (NmnEthernetItem *item)
{
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnEthernetItemPrivate *priv;
    NMDevice *device;

    object = G_OBJECT_CLASS (nmn_ethernet_item_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);
    device = nmn_network_item_get_device (NMN_NETWORK_ITEM (object));
    priv->state_changed_id = g_signal_connect (device, "state-changed", G_CALLBACK (state_changed), object);

    state_changed (device, nm_device_get_state (device), 
                   NM_DEVICE_STATE_UNKNOWN, NM_DEVICE_STATE_REASON_NONE, object);

    return object;
}

static void
dispose (GObject *object)
{
    NmnEthernetItemPrivate *priv = GET_PRIVATE (object);
    NMDevice *device;

    if (priv->disposed)
        return;

    device = nmn_network_item_get_device (NMN_NETWORK_ITEM (object));
    g_signal_handler_disconnect (device, priv->state_changed_id);

    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_ethernet_item_parent_class)->dispose (object);
}

static void
nmn_ethernet_item_class_init (NmnEthernetItemClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    NmnItemClass *item_class = NMN_ITEM_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnEthernetItemPrivate));

    object_class->constructor = constructor;
    object_class->dispose = dispose;

    item_class->connect = connect;
    item_class->disconnect = disconnect;
    item_class->get_priority = get_priority;
}
