/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <dbus/dbus-glib-lowlevel.h>
#include <nbtk-gtk-light-switch.h>
#include "nmn-applet.h"
#include "nmn-nm-data.h"
#include "nmn-status-icon.h"
#include "nmn-plug.h"
#include "nmn-networks.h"
#include "nmn-new-connection.h"

G_DEFINE_TYPE (NmnApplet, nmn_applet, G_TYPE_OBJECT)

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_APPLET, NmnAppletPrivate))

typedef struct {
    NmnNMData *nm_data;
    GtkStatusIcon *status_icon;
    GtkBuilder *builder;

    GtkWidget *plug;
    GtkWidget *pane;
    GtkWidget *list;
    GtkTable *switch_table;
    GtkWidget *enable_wifi;
    GtkWidget *enable_ethernet;
    GtkWidget *enable_3g;
    GtkWidget *enable_flightmode;
    GtkWidget *add_new_connection;

    GtkWidget *new_dialog;

    gboolean network_list_populated;
    int pop_counter;

    gboolean disposed;
} NmnAppletPrivate;

NmnApplet *
nmn_applet_new ()
{
    return NMN_APPLET (g_object_new (NMN_TYPE_APPLET, NULL));
}

GtkStatusIcon *
nmn_applet_get_status_icon (NmnApplet *applet)
{
    g_return_val_if_fail (NMN_IS_APPLET (applet), NULL);

    return GET_PRIVATE (applet)->status_icon;
}

GtkWidget *
nmn_applet_get_plug (NmnApplet *applet)
{
    g_return_val_if_fail (NMN_IS_APPLET (applet), NULL);

    return GET_PRIVATE (applet)->plug;
}

/* enable/disable wifi button */

static void
enable_wifi_toggled (NbtkGtkLightSwitch *w,
                     gboolean active,
                     gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    nmn_nm_data_wifi_toggled (priv->nm_data, active);
}

static void
wifi_toggled (NmnNMData *nm_data,
                  gboolean active,
                  gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    g_signal_handlers_block_by_func (priv->nm_data, enable_wifi_toggled, user_data);
    nbtk_gtk_light_switch_set_active (NBTK_GTK_LIGHT_SWITCH (priv->enable_wifi), active);
    g_signal_handlers_unblock_by_func (priv->nm_data, enable_wifi_toggled, user_data);
}

static void
enable_wifi_setup (NmnApplet *applet)
{
    NmnAppletPrivate *priv = GET_PRIVATE (applet);

    priv->enable_wifi = nbtk_gtk_light_switch_new ();
    gtk_table_attach (priv->switch_table,
                      priv->enable_wifi,
                      1, 2, 0, 1, 
                      GTK_FILL, 0,
                      0, 0);

    g_signal_connect (priv->enable_wifi, "switch-flipped", G_CALLBACK (enable_wifi_toggled), applet);
    gtk_widget_show (priv->enable_wifi);

    g_signal_connect (priv->nm_data, "wifi-toggled", G_CALLBACK (wifi_toggled), applet);
    wifi_toggled (priv->nm_data, nmn_nm_data_wifi_get_active (priv->nm_data), applet);
}

/* enable/disable ethernet button */

static void
enable_ethernet_toggled (NbtkGtkLightSwitch *w,
                         gboolean active,
                         gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    nmn_nm_data_ethernet_toggled (priv->nm_data, active);
}

static void
ethernet_toggled (NmnNMData *nm_data,
                  gboolean active,
                  gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    g_signal_handlers_block_by_func (priv->nm_data, enable_ethernet_toggled, user_data);
    nbtk_gtk_light_switch_set_active (NBTK_GTK_LIGHT_SWITCH (priv->enable_ethernet), active);
    g_signal_handlers_unblock_by_func (priv->nm_data, enable_ethernet_toggled, user_data);
}

static void
enable_ethernet_setup (NmnApplet *applet)
{
    NmnAppletPrivate *priv = GET_PRIVATE (applet);

    priv->enable_ethernet = nbtk_gtk_light_switch_new ();
    gtk_table_attach (priv->switch_table,
                      priv->enable_ethernet,
                      1, 2, 1, 2, 
                      GTK_FILL, 0,
                      0, 0);
 
    g_signal_connect (priv->enable_ethernet, "switch-flipped", G_CALLBACK (enable_ethernet_toggled), applet);
    gtk_widget_show (priv->enable_ethernet);

    g_signal_connect (priv->nm_data, "ethernet-toggled", G_CALLBACK (ethernet_toggled), applet);
    ethernet_toggled (priv->nm_data, nmn_nm_data_ethernet_get_active (priv->nm_data), applet);
}

/* enable/disable 3G button */

static void
enable_3g_toggled (NbtkGtkLightSwitch *w,
                   gboolean active,
                   gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    nmn_nm_data_modems_toggled (priv->nm_data, active);
}

static void
modems_toggled (NmnNMData *nm_data,
                gboolean active,
                gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    g_signal_handlers_block_by_func (priv->nm_data, enable_3g_toggled, user_data);
    nbtk_gtk_light_switch_set_active (NBTK_GTK_LIGHT_SWITCH (priv->enable_3g), active);
    g_signal_handlers_unblock_by_func (priv->nm_data, enable_3g_toggled, user_data);
}

static void
enable_3g_setup (NmnApplet *applet)
{
    NmnAppletPrivate *priv = GET_PRIVATE (applet);

    priv->enable_3g = nbtk_gtk_light_switch_new ();
    gtk_table_attach (priv->switch_table,
                      priv->enable_3g,
                      1, 2, 2, 3, 
                      GTK_FILL, 0,
                      0, 0);
    g_signal_connect (priv->enable_3g, "switch-flipped", G_CALLBACK (enable_3g_toggled), applet);
    gtk_widget_show (priv->enable_3g);

    g_signal_connect (priv->nm_data, "modems-toggled", G_CALLBACK (modems_toggled), applet);
    modems_toggled (priv->nm_data, nmn_nm_data_modems_get_active (priv->nm_data), applet);
}

/* enable/disable Fligh mode button */

static void
enable_flightmode_toggled (NbtkGtkLightSwitch *w,
                           gboolean active,
                           gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    gtk_widget_set_sensitive (priv->enable_wifi, !active);
    gtk_widget_set_sensitive (priv->enable_3g, !active);

    nmn_nm_data_flight_mode_toggled (priv->nm_data, active);
}

static void
flightmode_toggled (NmnNMData *nm_data,
                    gboolean active,
                    gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    g_signal_handlers_block_by_func (priv->nm_data, enable_flightmode_toggled, user_data);
    nbtk_gtk_light_switch_set_active (NBTK_GTK_LIGHT_SWITCH (priv->enable_flightmode), active);
    g_signal_handlers_unblock_by_func (priv->nm_data, enable_flightmode_toggled, user_data);

    gtk_widget_set_sensitive (priv->enable_flightmode, nmn_nm_data_flight_mode_can_change (priv->nm_data));
    gtk_widget_set_sensitive (priv->enable_3g, !active);
    gtk_widget_set_sensitive (priv->enable_wifi, !active);
}

static void
enable_flightmode_setup (NmnApplet *applet)
{
    NmnAppletPrivate *priv = GET_PRIVATE (applet);

    priv->enable_flightmode = nbtk_gtk_light_switch_new ();
    gtk_table_attach (priv->switch_table,
                      priv->enable_flightmode,
                      1, 2, 4, 5, 
                      GTK_FILL, 0,
                      0, 0);

    g_signal_connect (priv->enable_flightmode, "switch-flipped", G_CALLBACK (enable_flightmode_toggled), applet);
    gtk_widget_show (priv->enable_flightmode);

    g_signal_connect (priv->nm_data, "flight-mode-toggled", G_CALLBACK (flightmode_toggled), applet);
    flightmode_toggled (priv->nm_data, nmn_nm_data_flight_mode_get_active (priv->nm_data), applet);
}

/* add new connection button */

static void
add_new_connection_show (GtkButton *button,
                         gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    gtk_widget_show (priv->new_dialog);
    nmn_plug_push (NMN_PLUG (priv->plug), priv->new_dialog);
    priv->pop_counter++;
}

static void
add_new_connection_hide (GtkWidget *widget,
                         gpointer user_data)
{
    NmnAppletPrivate *priv = GET_PRIVATE (user_data);

    nmn_plug_pop (NMN_PLUG (priv->plug));
    priv->pop_counter--;
}

static void
add_new_connection_setup (NmnApplet *applet)
{
    NmnAppletPrivate *priv = GET_PRIVATE (applet);

    priv->new_dialog = nmn_new_connection_create (priv->builder, priv->nm_data);
    g_object_ref_sink (priv->new_dialog);
    g_signal_connect (priv->new_dialog, "hide", G_CALLBACK (add_new_connection_hide), applet);

    priv->add_new_connection = GTK_WIDGET (gtk_builder_get_object (priv->builder, "add_new_connection"));
    g_signal_connect (priv->add_new_connection, "clicked", G_CALLBACK (add_new_connection_show), applet);
}

void
nmn_applet_set_visible (NmnApplet *applet,
                        gboolean visible)
{
    NmnAppletPrivate *priv = GET_PRIVATE (applet);

    nmn_status_icon_set_active (NMN_STATUS_ICON (priv->status_icon), visible);
    
    if (visible) {
        if (!priv->network_list_populated) {
            nmn_networks_populate (NMN_NETWORKS (priv->list));
            priv->network_list_populated = TRUE;
        }
    } else {
        while (priv->pop_counter > 0) {
            nmn_plug_pop (NMN_PLUG (priv->plug));
            priv->pop_counter--;
        }
    }
}

static DBusGConnection *
init_dbus (void)
{
    DBusGConnection *bus;
    DBusGProxy *proxy;
    GError *error = NULL;
    int request_name_result;

    dbus_connection_set_change_sigpipe (TRUE);

    bus = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
    if (!bus) {
        g_warning ("Could not get the system bus.  Make sure "
                   "the message bus daemon is running!  Message: %s",
                   error->message);
        g_error_free (error);
        return NULL;
    }

    dbus_connection_set_exit_on_disconnect (dbus_g_connection_get_connection (bus), FALSE);

    proxy = dbus_g_proxy_new_for_name (bus,
                                       "org.freedesktop.DBus",
                                       "/org/freedesktop/DBus",
                                       "org.freedesktop.DBus");

    if (!dbus_g_proxy_call (proxy, "RequestName", &error,
                            G_TYPE_STRING, NM_DBUS_SERVICE_USER_SETTINGS,
                            G_TYPE_UINT, DBUS_NAME_FLAG_DO_NOT_QUEUE,
                            G_TYPE_INVALID,
                            G_TYPE_UINT, &request_name_result,
                            G_TYPE_INVALID)) {
        g_warning ("Could not acquire the NetworkManagerUserSettings service.\n"
                   "  Message: '%s'", error->message);
        g_error_free (error);
        goto err;
    }

    if (request_name_result != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
        g_warning ("Could not acquire the NetworkManagerUserSettings service "
                   "as it is already taken.  Return: %d",
                   request_name_result);
        goto err;
    }

    return bus;

 err:
    dbus_g_connection_unref (bus);
    return NULL;
}

static void
nmn_applet_init (NmnApplet *applet)
{
    NmnAppletPrivate *priv = GET_PRIVATE (applet);
    DBusGConnection *bus;
    GtkWidget *container;

    bus = init_dbus ();
    /* FIXME: Do something nicer here */
    g_assert (bus);

    priv->nm_data = nmn_nm_data_new (bus);
    dbus_g_connection_unref (bus);

    priv->status_icon = nmn_status_icon_new ();

    dbus_g_connection_register_g_object (nm_object_get_connection (NM_OBJECT (priv->nm_data)),
	                                     NM_DBUS_PATH_SETTINGS,
	                                     G_OBJECT (nmn_nm_data_get_user_settings (priv->nm_data)));

    nmn_status_icon_set_client (NMN_STATUS_ICON (priv->status_icon), NM_CLIENT (priv->nm_data));

    priv->builder = gtk_builder_new ();
    gtk_builder_add_from_file (priv->builder, UIDIR "/network-manager-netbook.ui", NULL);

    priv->pane = GTK_WIDGET (gtk_builder_get_object (priv->builder, "main_container"));
    priv->list = nmn_networks_new (priv->nm_data);

    container = GTK_WIDGET (gtk_builder_get_object (priv->builder, "networks_container"));
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (container), priv->list);
    gtk_widget_show (priv->list);

    priv->switch_table = GTK_TABLE (gtk_builder_get_object (priv->builder, "switch_table"));

    enable_wifi_setup (applet);
    enable_ethernet_setup (applet);
    enable_3g_setup (applet);
    enable_flightmode_setup (applet);
    add_new_connection_setup (applet);

    priv->plug = nmn_plug_new ();
    g_object_ref (priv->pane);
    gtk_widget_unparent (priv->pane);
    nmn_plug_push (NMN_PLUG (priv->plug), priv->pane);
    g_object_unref (priv->pane);
}

static void
dispose (GObject *object)
{
    NmnAppletPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    priv->disposed = TRUE;

    if (priv->new_dialog)
        gtk_widget_destroy (priv->new_dialog);

    if (priv->builder)
        g_object_unref (priv->builder);

    if (priv->status_icon)
        g_object_unref (priv->status_icon);

    if (priv->nm_data)
        g_object_unref (priv->nm_data);

    if (priv->plug)
        g_object_unref (priv->plug);

    G_OBJECT_CLASS (nmn_applet_parent_class)->dispose (object);
}

static void
nmn_applet_class_init (NmnAppletClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnAppletPrivate));

    object_class->dispose = dispose;
}
